<?php
class AuthController extends Controller {
    private $userModel;

    public function __construct() {
        parent::__construct();
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $this->userModel = new User();
    }

    public function login() {
        // Si l'utilisateur est déjà connecté, rediriger vers la page d'accueil
        if ($this->isLoggedIn()) {
            if ($_SESSION['user_role'] === 'admin') {
                $this->redirect('admin');
            } else {
                $this->redirect('');
            }
        }

        if ($this->isPost()) {
            $email = trim($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';

            // Vérifier les identifiants
            $result = $this->userModel->login($email, $password);

            if ($result['success']) {
                // Connecter l'utilisateur et rediriger vers la page appropriée
                $_SESSION['user_id'] = $result['user']['id'];
                $_SESSION['user_name'] = $result['user']['first_name'] . ' ' . $result['user']['last_name'];
                $_SESSION['user_email'] = $result['user']['email'];
                $_SESSION['user_role'] = $result['user']['role'];

                $_SESSION['success'] = 'Connexion réussie. Bienvenue ' . $_SESSION['user_name'] . ' !';

                // Rediriger vers la page d'administration si l'utilisateur est admin
                if ($_SESSION['user_role'] === 'admin') {
                    $this->redirect('admin');
                } else {
                    $this->redirect('diagnostic');
                }
            } else {
                $_SESSION['error'] = $result['message'];
                $this->renderAuth('auth/login', [
                    'title' => 'Connexion',
                    'email' => $email
                ]);
            }
        } else {
            $this->renderAuth('auth/login', [
                'title' => 'Connexion'
            ]);
        }
    }

    public function register() {
        // Si l'utilisateur est déjà connecté, rediriger vers la page d'accueil
        if ($this->isLoggedIn()) {
            if ($_SESSION['user_role'] === 'admin') {
                $this->redirect('admin');
            } else {
                $this->redirect('');
            }
        }

        if ($this->isPost()) {
            // Récupérer les données du formulaire
            $firstName = trim($_POST['first_name'] ?? '');
            $lastName = trim($_POST['last_name'] ?? '');
            $email = trim($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';
            $confirmPassword = $_POST['confirm_password'] ?? '';

            // Valider les données
            $errors = [];

            if (empty($firstName)) {
                $errors[] = 'Le prénom est requis';
            }

            if (empty($lastName)) {
                $errors[] = 'Le nom est requis';
            }

            if (empty($email)) {
                $errors[] = 'L\'email est requis';
            } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'L\'email n\'est pas valide';
            }

            if (empty($password)) {
                $errors[] = 'Le mot de passe est requis';
            } elseif (strlen($password) < 6) {
                $errors[] = 'Le mot de passe doit contenir au moins 6 caractères';
            }

            if ($password !== $confirmPassword) {
                $errors[] = 'Les mots de passe ne correspondent pas';
            }

            if (empty($errors)) {
                // Créer l'utilisateur
                $result = $this->userModel->register($email, $password, $firstName, $lastName);

                if ($result['success']) {
                    $_SESSION['success'] = $result['message'];
                    $this->redirect('auth/login');
                } else {
                    $_SESSION['error'] = $result['message'];
                    $this->renderAuth('auth/register', [
                        'title' => 'Inscription',
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'email' => $email
                    ]);
                }
            } else {
                $_SESSION['error'] = implode('<br>', $errors);
                $this->renderAuth('auth/register', [
                    'title' => 'Inscription',
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                    'email' => $email
                ]);
            }
        } else {
            $this->renderAuth('auth/register', [
                'title' => 'Inscription'
            ]);
        }
    }

    public function logout() {
        session_destroy();
        $this->redirect('auth/login');
    }

    public function profile() {
        // Vérifier si l'utilisateur est connecté
        $this->requireLogin();

        $user = $this->userModel->getUserById($_SESSION['user_id']);
        
        if ($this->isPost()) {
            $data = [
                'first_name' => $_POST['first_name'] ?? $user['first_name'],
                'last_name' => $_POST['last_name'] ?? $user['last_name'],
                'email' => $_POST['email'] ?? $user['email'],
                'current_password' => $_POST['current_password'] ?? '',
                'new_password' => $_POST['new_password'] ?? ''
            ];
            
            $result = $this->userModel->updateProfile($_SESSION['user_id'], $data);
            
            if ($result['success']) {
                $_SESSION['success'] = $result['message'];
                $_SESSION['user_name'] = $data['first_name'] . ' ' . $data['last_name'];
                $_SESSION['user_email'] = $data['email'];
                $this->redirect('auth/profile');
            } else {
                $_SESSION['error'] = $result['message'];
                $this->redirect('auth/profile');
            }
        }
        
        $this->render('auth/profile', [
            'title' => 'Mon profil',
            'user' => $user
        ]);
    }
} 