<?php
class DiagnosticController extends Controller {
    private $symptomModel;
    private $diagnosticModel;

    public function __construct() {
        parent::__construct();
        
        // Vérifier si l'utilisateur est connecté pour toutes les actions du contrôleur
        $this->requireLogin();
        
        // Rediriger l'admin vers son dashboard
        if ($this->isLoggedIn() && isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin') {
            $this->redirect('admin');
        }
        
        $this->symptomModel = new Symptom();
        $this->diagnosticModel = new Diagnostic();
    }

    public function index() {
        // Récupérer les symptômes de la base de données
        $symptoms = $this->symptomModel->getSymptomsByCategory();
        $this->render('diagnostic/index', [
            'title' => 'Nouveau diagnostic',
            'symptoms' => $symptoms
        ]);
    }

    public function analyze() {
        if (!$this->isPost()) {
            $this->redirect('diagnostic');
        }

        $selectedSymptoms = isset($_POST['symptoms']) ? $_POST['symptoms'] : [];
        
        // Check if symptoms is a string with comma separated values and convert to array
        if (is_string($selectedSymptoms) && strpos($selectedSymptoms, ',') !== false) {
            $selectedSymptoms = explode(',', $selectedSymptoms);
        }
        
        // S'assurer que $selectedSymptoms est un tableau
        if (!is_array($selectedSymptoms)) {
            $selectedSymptoms = [$selectedSymptoms];
        }
        
        if (empty($selectedSymptoms)) {
            $_SESSION['error'] = 'Veuillez sélectionner au moins un symptôme.';
            $this->redirect('diagnostic');
        }

        // Analyser les symptômes
        $results = $this->diagnosticModel->analyzeSymptoms($selectedSymptoms);
        
        // Sauvegarder le diagnostic
        $this->diagnosticModel->saveDiagnostic($_SESSION['user_id'], $selectedSymptoms, $results);
        
        // Stocker les résultats dans la session pour l'affichage
        $_SESSION['diagnostic_results'] = $results;
        $_SESSION['selected_symptoms'] = $selectedSymptoms;
        
        $this->redirect('diagnostic/results');
    }

    public function results() {
        if (!isset($_SESSION['diagnostic_results'])) {
            $this->redirect('diagnostic');
        }
        
        $results = $_SESSION['diagnostic_results'];
        $selectedSymptomIds = $_SESSION['selected_symptoms'];
        
        // Récupérer les noms des symptômes au lieu des IDs
        $symptomNames = [];
        foreach ($selectedSymptomIds as $symptomId) {
            $symptom = $this->symptomModel->getSymptomById($symptomId);
            if ($symptom) {
                $symptomNames[$symptomId] = $symptom['name'];
            }
        }
        
        $this->render('diagnostic/results', [
            'title' => 'Résultats du diagnostic',
            'results' => $results,
            'selectedSymptoms' => $symptomNames
        ]);
    }
    
    public function history() {
        // Check if we're searching
        $query = isset($_GET['search']) ? trim($_GET['search']) : '';
        
        if (!empty($query)) {
            // Search for diagnostics
            $diagnostics = $this->diagnosticModel->searchUserDiagnostics($_SESSION['user_id'], $query);
        } else {
            // Get all diagnostics
            $diagnostics = $this->diagnosticModel->getUserDiagnostics($_SESSION['user_id']);
        }
        
        // Add pagination
        $itemsPerPage = 10;
        $currentPage = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $totalItems = count($diagnostics);
        $totalPages = ceil($totalItems / $itemsPerPage);
        
        // Ensure current page is valid
        if ($currentPage < 1) $currentPage = 1;
        if ($currentPage > $totalPages && $totalPages > 0) $currentPage = $totalPages;
        
        // Slice the array for pagination
        $start = ($currentPage - 1) * $itemsPerPage;
        $paginatedDiagnostics = array_slice($diagnostics, $start, $itemsPerPage);
        
        $this->render('diagnostic/history', [
            'title' => 'Historique des diagnostics',
            'diagnostics' => $paginatedDiagnostics,
            'currentPage' => $currentPage,
            'totalPages' => $totalPages,
            'query' => $query
        ]);
    }
    
    public function details($id) {
        $diagnostic = $this->diagnosticModel->getDiagnosticById($id, $_SESSION['user_id']);
        
        if (!$diagnostic) {
            $_SESSION['error'] = 'Diagnostic non trouvé.';
            $this->redirect('diagnostic/history');
        }
        
        $this->render('diagnostic/details', [
            'title' => 'Détails du diagnostic',
            'diagnostic' => $diagnostic
        ]);
    }
    
    public function search() {
        $query = isset($_GET['q']) ? $_GET['q'] : '';
        $diagnostics = [];
        
        if (!empty($query)) {
            $diagnostics = $this->diagnosticModel->searchUserDiagnostics($_SESSION['user_id'], $query);
        }
        
        $this->render('diagnostic/search', [
            'title' => 'Recherche de diagnostics',
            'query' => $query,
            'diagnostics' => $diagnostics
        ]);
    }
} 