<?php

class Controller {
    public function __construct() {
        // Démarrer la session si elle n'est pas déjà démarrée
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }

    protected function render($view, $data = []) {
        // Extraire les données pour les rendre disponibles dans la vue
        extract($data);
        
        // Déterminer quel layout utiliser
        $viewParts = explode('/', $view);
        $layoutType = $viewParts[0]; // admin, auth, diagnostic, etc.
        
        // Définir les layouts disponibles et leur correspondance
        $layouts = [
            'admin' => 'admin',
            'auth' => 'auth',
            'diagnostic' => 'main',
            'home' => 'main',
            'errors' => 'main'
        ];
        
        // Définir le chemin vers la vue
        $viewFile = APP_ROOT . '/views/' . $view . '.php';
        
        // Si la vue existe
        if (file_exists($viewFile)) {
            // Démarrer la mise en mémoire tampon
            ob_start();
            
            // Charger la vue
            require_once $viewFile;
            
            // Récupérer le contenu de la vue
            $content = ob_get_clean();
            
            // Choix du layout
            $layout = $layouts[$layoutType] ?? 'main';
            
            // Charger le layout avec le contenu
            $layoutFile = APP_ROOT . '/views/layouts/' . $layout . '.php';
            if (file_exists($layoutFile)) {
                require_once $layoutFile;
            } else {
                // Fallback au layout principal si le layout spécifique n'existe pas
                require_once APP_ROOT . '/views/layouts/main.php';
            }
        } else {
            // Si la vue n'existe pas, afficher une erreur 404
            header("HTTP/1.0 404 Not Found");
            require_once APP_ROOT . '/views/errors/404.php';
        }
    }
    
    // Rendu simplifié sans menu de navigation pour les pages d'authentification
    protected function renderAuth($view, $data = []) {
        // Extraire les données pour les rendre disponibles dans la vue
        extract($data);
        
        // Définir le chemin vers la vue
        $viewFile = APP_ROOT . '/views/' . $view . '.php';
        
        // Si la vue existe
        if (file_exists($viewFile)) {
            // Définir le titre de la page s'il n'est pas défini
            $title = $data['title'] ?? 'Authentification';
            
            // Afficher la page HTML de base
            echo '<!DOCTYPE html>
            <html lang="fr">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>' . $title . ' - ' . APP_NAME . '</title>
                <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
                <style>
                    body {
                        font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif;
                        background-color: #f5f5f5;
                        margin: 0;
                        padding: 0;
                        display: flex;
                        justify-content: center;
                        align-items: center;
                        min-height: 100vh;
                    }
                    .auth-container {
                        max-width: 500px;
                        width: 100%;
                        padding: 20px;
                    }
                    .logo {
                        text-align: center;
                        margin-bottom: 20px;
                    }
                    .logo a {
                        font-size: 2rem;
                        font-weight: bold;
                        color: #007bff;
                        text-decoration: none;
                    }
                    .alert {
                        padding: 15px;
                        margin-bottom: 20px;
                        border-radius: 4px;
                    }
                    .alert-success {
                        background-color: #d4edda;
                        color: #155724;
                        border: 1px solid #c3e6cb;
                    }
                    .alert-danger {
                        background-color: #f8d7da;
                        color: #721c24;
                        border: 1px solid #f5c6cb;
                    }
                </style>
            </head>
            <body>
                <div class="auth-container">
                    <div class="logo">
                        <a href="' . APP_URL . '">' . APP_NAME . '</a>
                    </div>';
                    
            // Afficher les alertes le cas échéant
            if (isset($_SESSION['success'])) {
                echo '<div class="alert alert-success">' . $_SESSION['success'] . '</div>';
                unset($_SESSION['success']);
            }
            
            if (isset($_SESSION['error'])) {
                echo '<div class="alert alert-danger">' . $_SESSION['error'] . '</div>';
                unset($_SESSION['error']);
            }
            
            // Inclure la vue
            include $viewFile;
            
            // Fermer la page HTML
            echo '</div>
            </body>
            </html>';
        } else {
            // Si la vue n'existe pas, afficher une erreur 404
            header("HTTP/1.0 404 Not Found");
            echo "Page not found";
        }
    }
    
    protected function redirect($url) {
        // Supprimer le slash initial si présent pour éviter les doubles slashes
        $url = ltrim($url, '/');
        
        // S'il s'agit d'une URL vide, rediriger vers la racine de l'application
        if (empty($url)) {
            header('Location: ' . APP_URL);
        } else {
            header('Location: ' . APP_URL . '/' . $url);
        }
        exit;
    }
    
    protected function isPost() {
        return $_SERVER['REQUEST_METHOD'] === 'POST';
    }
    
    protected function isGet() {
        return $_SERVER['REQUEST_METHOD'] === 'GET';
    }
    
    protected function getPostData() {
        return $_POST;
    }
    
    protected function getQueryData() {
        return $_GET;
    }
    
    protected function isLoggedIn() {
        return isset($_SESSION['user_id']);
    }
    
    protected function requireLogin() {
        if (!$this->isLoggedIn()) {
            $_SESSION['error'] = 'Vous devez être connecté pour accéder à cette page.';
            $this->redirect('auth/login');
        }
    }
    
    protected function requireAdmin() {
        $this->requireLogin();
        
        if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'admin') {
            $_SESSION['error'] = 'Vous n\'avez pas les droits pour accéder à cette page.';
            $this->redirect('home');
        }
    }
    
    protected function redirectIfAdmin() {
        if ($this->isLoggedIn() && isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin') {
            $this->redirect('admin');
        }
    }
    
    protected function redirectIfUser() {
        if ($this->isLoggedIn() && isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'user') {
            $this->redirect('home');
        }
    }
} 