<?php
class Symptom {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function getAllSymptoms() {
        try {
            $stmt = $this->db->query("SELECT * FROM symptoms ORDER BY name");
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error in getAllSymptoms: " . $e->getMessage());
            return [];
        }
    }

    public function getSymptomById($id) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM symptoms WHERE id = ?");
            $stmt->execute([$id]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Error in getSymptomById: " . $e->getMessage());
            return false;
        }
    }

    public function getSymptomsByCategory() {
        try {
            $stmt = $this->db->query("SELECT * FROM symptoms ORDER BY category, name");
            $symptoms = $stmt->fetchAll();
            
            $categorized = [];
            foreach ($symptoms as $symptom) {
                $categorized[$symptom['category']][] = $symptom;
            }
            
            return $categorized;
        } catch (PDOException $e) {
            error_log("Error in getSymptomsByCategory: " . $e->getMessage());
            return [];
        }
    }

    public function countSymptoms() {
        try {
            $stmt = $this->db->query("SELECT COUNT(*) as count FROM symptoms");
            $result = $stmt->fetch();
            return $result['count'];
        } catch (PDOException $e) {
            error_log("Error in countSymptoms: " . $e->getMessage());
            return 0;
        }
    }

    public function addSymptom($name, $description, $category) {
        try {
            // Vérifier si le symptôme existe déjà
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM symptoms WHERE name = ?");
            $stmt->execute([$name]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] > 0) {
                return [
                    'success' => false,
                    'message' => 'Un symptôme avec ce nom existe déjà.'
                ];
            }
            
            $stmt = $this->db->prepare("INSERT INTO symptoms (name, description, category) VALUES (?, ?, ?)");
            $success = $stmt->execute([$name, $description, $category]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'Le symptôme a été ajouté avec succès.',
                    'id' => $this->db->lastInsertId()
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de l\'ajout du symptôme.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in addSymptom: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }

    public function updateSymptom($id, $name, $description, $category) {
        try {
            // Vérifier si le symptôme existe
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM symptoms WHERE id = ?");
            $stmt->execute([$id]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] == 0) {
                return [
                    'success' => false,
                    'message' => 'Le symptôme n\'existe pas.'
                ];
            }
            
            // Vérifier si le nom existe déjà pour un autre symptôme
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM symptoms WHERE name = ? AND id != ?");
            $stmt->execute([$name, $id]);
            $nameExists = $stmt->fetch();
            
            if ($nameExists['count'] > 0) {
                return [
                    'success' => false,
                    'message' => 'Un autre symptôme avec ce nom existe déjà.'
                ];
            }
            
            $stmt = $this->db->prepare("UPDATE symptoms SET name = ?, description = ?, category = ? WHERE id = ?");
            $success = $stmt->execute([$name, $description, $category, $id]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'Le symptôme a été mis à jour avec succès.'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de la mise à jour du symptôme.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in updateSymptom: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }

    public function deleteSymptom($id) {
        try {
            // Vérifier si le symptôme existe
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM symptoms WHERE id = ?");
            $stmt->execute([$id]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] == 0) {
                return [
                    'success' => false,
                    'message' => 'Le symptôme n\'existe pas.'
                ];
            }
            
            // Vérifier si le symptôme est utilisé dans des relations
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM disease_symptom WHERE symptom_id = ?");
            $stmt->execute([$id]);
            $hasRelations = $stmt->fetch();
            
            if ($hasRelations['count'] > 0) {
                return [
                    'success' => false,
                    'message' => 'Ce symptôme est utilisé dans des relations avec des maladies. Veuillez d\'abord supprimer ces relations.'
                ];
            }
            
            $stmt = $this->db->prepare("DELETE FROM symptoms WHERE id = ?");
            $success = $stmt->execute([$id]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'Le symptôme a été supprimé avec succès.'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de la suppression du symptôme.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in deleteSymptom: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }
} 