<?php
$title = 'Historique des Diagnostics';
?>

<style>
    :root {
        --primary-color: #007bff;
        --secondary-color: #6c757d;
        --success-color: #28a745;
        --warning-color: #ffc107;
        --danger-color: #dc3545;
        --light-color: #f8f9fa;
        --border-color: #dee2e6;
    }

    .container {
        max-width: 1000px;
        margin: 2rem auto;
        padding: 0 1rem;
    }

    .history-card {
        background: white;
        border-radius: 8px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        padding: 2rem;
        margin-bottom: 2rem;
    }

    .history-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 2rem;
        flex-wrap: wrap;
    }

    .history-header h2 {
        color: var(--primary-color);
        margin: 0;
    }

    .search-container {
        display: flex;
        gap: 0.5rem;
        margin-top: 1rem;
    }

    .search-input {
        padding: 0.5rem;
        border: 1px solid var(--border-color);
        border-radius: 4px;
        min-width: 250px;
    }

    .btn {
        display: inline-flex;
        align-items: center;
        padding: 0.5rem 1rem;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        text-decoration: none;
        font-weight: 500;
        transition: background-color 0.3s;
    }

    .btn i {
        margin-right: 0.5rem;
    }

    .btn-primary {
        background-color: var(--primary-color);
        color: white;
    }

    .btn-primary:hover {
        background-color: #0056b3;
    }

    .btn-secondary {
        background-color: var(--secondary-color);
        color: white;
    }

    .btn-secondary:hover {
        background-color: #5a6268;
    }

    .history-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 1rem;
    }

    .history-table th, 
    .history-table td {
        padding: 0.75rem;
        text-align: left;
        border-bottom: 1px solid var(--border-color);
    }

    .history-table th {
        background-color: var(--light-color);
        font-weight: 600;
        color: var(--secondary-color);
    }

    .history-table tr:hover {
        background-color: rgba(0, 123, 255, 0.05);
    }

    .symptoms-cell {
        max-width: 300px;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    }

    .disease-cell {
        font-weight: 500;
    }

    .probability {
        display: inline-block;
        padding: 0.25rem 0.5rem;
        border-radius: 20px;
        font-size: 0.85rem;
        font-weight: 600;
        text-align: center;
        min-width: 60px;
    }

    .probability-high {
        background-color: var(--danger-color);
        color: white;
    }

    .probability-medium {
        background-color: var(--warning-color);
        color: #212529;
    }

    .probability-low {
        background-color: var(--success-color);
        color: white;
    }

    .action-links {
        display: flex;
        gap: 0.5rem;
    }

    .action-link {
        color: var(--primary-color);
        text-decoration: none;
    }

    .action-link:hover {
        text-decoration: underline;
    }

    .empty-history {
        text-align: center;
        padding: 2rem;
        color: var(--secondary-color);
    }

    .pagination {
        display: flex;
        justify-content: center;
        list-style: none;
        padding: 0;
        margin-top: 2rem;
    }

    .page-item {
        margin: 0 0.25rem;
    }

    .page-link {
        display: block;
        padding: 0.5rem 0.75rem;
        border: 1px solid var(--border-color);
        border-radius: 4px;
        color: var(--primary-color);
        text-decoration: none;
    }

    .page-link:hover {
        background-color: var(--light-color);
    }

    .page-item.active .page-link {
        background-color: var(--primary-color);
        color: white;
        border-color: var(--primary-color);
    }

    @media (max-width: 768px) {
        .history-header {
            flex-direction: column;
            align-items: flex-start;
        }

        .search-container {
            width: 100%;
            margin-top: 1rem;
        }

        .search-input {
            flex-grow: 1;
        }

        .history-table, 
        .history-table thead, 
        .history-table tbody, 
        .history-table th, 
        .history-table td, 
        .history-table tr {
            display: block;
        }

        .history-table thead tr {
            position: absolute;
            top: -9999px;
            left: -9999px;
        }

        .history-table tr {
            border: 1px solid var(--border-color);
            margin-bottom: 1rem;
            border-radius: 4px;
        }

        .history-table td {
            border: none;
            border-bottom: 1px solid var(--border-color);
            position: relative;
            padding-left: 50%;
        }

        .history-table td:before {
            position: absolute;
            top: 12px;
            left: 6px;
            width: 45%;
            padding-right: 10px;
            white-space: nowrap;
            font-weight: 600;
        }

        .history-table td:nth-of-type(1):before { content: "Date"; }
        .history-table td:nth-of-type(2):before { content: "Symptômes"; }
        .history-table td:nth-of-type(3):before { content: "Maladie"; }
        .history-table td:nth-of-type(4):before { content: "Probabilité"; }
        .history-table td:nth-of-type(5):before { content: "Actions"; }
    }
</style>

<div class="container">
    <div class="history-card">
        <div class="history-header">
            <h2>Historique des Diagnostics</h2>
            <form action="<?= APP_URL ?>/diagnostic/history" method="GET" class="search-container">
                <input type="text" name="search" placeholder="Rechercher..." class="search-input" value="<?= isset($_GET['search']) ? htmlspecialchars($_GET['search']) : '' ?>">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-search"></i>
                    Rechercher
                </button>
                <?php if (isset($_GET['search'])): ?>
                <a href="<?= APP_URL ?>/diagnostic/history" class="btn btn-secondary">
                    <i class="fas fa-times"></i>
                    Effacer
                </a>
                <?php endif; ?>
            </form>
        </div>
        
        <?php if (empty($diagnostics)): ?>
            <div class="empty-history">
                <i class="fas fa-inbox fa-3x"></i>
                <p>Vous n'avez pas encore de diagnostics enregistrés.</p>
                <a href="<?= APP_URL ?>/diagnostic" class="btn btn-primary">
                    <i class="fas fa-stethoscope"></i>
                    Faire un diagnostic
                </a>
            </div>
        <?php else: ?>
            <table class="history-table">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Symptômes</th>
                        <th>Maladie</th>
                        <th>Probabilité</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($diagnostics as $diagnostic): ?>
                        <tr>
                            <td><?= date('d/m/Y H:i', strtotime($diagnostic['created_at'])) ?></td>
                            <td class="symptoms-cell" title="<?= htmlspecialchars($diagnostic['symptoms']) ?>">
                                <?= htmlspecialchars($diagnostic['symptoms']) ?>
                            </td>
                            <td class="disease-cell"><?= htmlspecialchars($diagnostic['top_disease'] ?? 'Non déterminé') ?></td>
                            <td>
                                <?php if (isset($diagnostic['top_probability'])): ?>
                                    <?php
                                    $probability = (float)$diagnostic['top_probability'];
                                    $probabilityClass = '';
                                    if ($probability >= 75) {
                                        $probabilityClass = 'probability-high';
                                    } elseif ($probability >= 50) {
                                        $probabilityClass = 'probability-medium';
                                    } else {
                                        $probabilityClass = 'probability-low';
                                    }
                                    ?>
                                    <span class="probability <?= $probabilityClass ?>">
                                        <?= number_format($probability, 1) ?>%
                                    </span>
                                <?php else: ?>
                                    <span>-</span>
                                <?php endif; ?>
                            </td>
                            <td class="action-links">
                                <a href="<?= APP_URL ?>/diagnostic/details/<?= $diagnostic['id'] ?>" class="action-link">
                                    <i class="fas fa-eye"></i> Détails
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <?php if ($totalPages > 1): ?>
                <ul class="pagination">
                    <?php if ($currentPage > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="<?= APP_URL ?>/diagnostic/history?page=<?= $currentPage - 1 ?><?= isset($_GET['search']) ? '&search=' . urlencode($_GET['search']) : '' ?>">
                                <i class="fas fa-chevron-left"></i>
                            </a>
                        </li>
                    <?php endif; ?>
                    
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                        <li class="page-item <?= $i == $currentPage ? 'active' : '' ?>">
                            <a class="page-link" href="<?= APP_URL ?>/diagnostic/history?page=<?= $i ?><?= isset($_GET['search']) ? '&search=' . urlencode($_GET['search']) : '' ?>">
                                <?= $i ?>
                            </a>
                        </li>
                    <?php endfor; ?>
                    
                    <?php if ($currentPage < $totalPages): ?>
                        <li class="page-item">
                            <a class="page-link" href="<?= APP_URL ?>/diagnostic/history?page=<?= $currentPage + 1 ?><?= isset($_GET['search']) ? '&search=' . urlencode($_GET['search']) : '' ?>">
                                <i class="fas fa-chevron-right"></i>
                            </a>
                        </li>
                    <?php endif; ?>
                </ul>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Gestion des dropdown menus dans la navbar
    const navbarDropdown = document.getElementById('navbarDropdown');
    const userDropdown = document.getElementById('userDropdown');
    
    if (navbarDropdown && userDropdown) {
        // Afficher/masquer le dropdown au clic
        navbarDropdown.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            userDropdown.classList.toggle('show');
        });
        
        // Fermer le dropdown si on clique ailleurs
        document.addEventListener('click', function(e) {
            if (!navbarDropdown.contains(e.target) && !userDropdown.contains(e.target)) {
                userDropdown.classList.remove('show');
            }
        });
    }
});
</script> 
